/**
 * \file: exchnd_backend_file.c
 *
 * Implementation of a backend that dumps exception handler
 * messages into a buffer.
 *
 * \component: exchndd
 *
 * \author: Kai Tomerius (ktomerius@de.adit-jv.com)
 *
 * \copyright (c) 2013 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * \see <related items>
 *
 * \history
 *
 ***********************************************************************/
#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <pthread.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "exchnd_backend.h"
#include "exchnd_interface.h"

/*
 * Implementation of an exception handler backend that dumps exception
 * messages to a file.
 */

typedef struct ExchndBackendFile {
    ExchndBackend_t base;     /* exception handler backend interface */
    FILE *fd;
} ExchndBackendFile_t;

/*
 * \func retrieve_msg
 *
 * Retrieve message from msg_list and store it in the backend.
 *
 * \param backend Pointer to the original backend.
 */
static void *retrieve_msg(void *backend)
{
    ExchndBackendFile_t *file_backend = (ExchndBackendFile_t *)backend;
    struct msg_list *current = mlist;
    char *msg = NULL;
    pthread_mutex_t *msg_lck = &mlist->msg_lck;

    pthread_mutex_lock(msg_lck);
    prctl(PR_SET_NAME, "be_file", NULL, NULL, NULL);

#if (DEFAULT_BACKEND & FILE_BACKEND)
    /* Default backend ready to receive datas. */
    pthread_cond_broadcast(&msg_handled);
#endif

    pthread_cond_wait(&new_msg, msg_lck);

    while (!(action & EXCHND_TERMINATE)) {
        current = mlist->next;

        while (current != mlist) {
            if (current->read_flag & FILE_BACKEND) {
                current = current->next;
                continue;
            }

            msg = current->msg;
            /* No need to keep the lock during write operation */
            pthread_mutex_unlock(msg_lck);
            file_backend->base.store(&file_backend->base,
                                     current->len, msg);

            /* Get lock again to update message structure */
            pthread_mutex_lock(msg_lck);
            current->read_flag |= FILE_BACKEND;
            current = current->next;
        }

#if (DEFAULT_BACKEND & FILE_BACKEND)
        /* List complete for default backend */
        pthread_cond_broadcast(&msg_handled);
#endif

        pthread_cond_wait(&new_msg, msg_lck);
    }

    pthread_mutex_unlock(msg_lck);

    return NULL;
}

/*
 * \func store
 *
 * Write a message to the file
 *
 * \param backend Pointer to backend structure (must be file!)
 * \param len Length of the data
 * \param msg Data to be written to the file
 *
 * \return 0 if successful, error code otherwise
 */
static unsigned int store(struct ExchndBackend *backend,
                          int len, char *msg)
{
    ExchndBackendFile_t *file_backend = (ExchndBackendFile_t *)backend;

    (void)len;

    if ((file_backend != NULL) && (file_backend->fd != NULL))
        fprintf(file_backend->fd, "%s", msg);
    else
        return EINVAL;

    return 0;
}

/*
 * \func destroy
 *
 * Destroy the given file backend structure and its contents. If the file
 * handle is not STDOUT it is closed.
 *
 * \param backend Pointer to backend structure (must be file!)
 */
static void destroy(struct ExchndBackend *backend)
{
    ExchndBackendFile_t *file_backend = (ExchndBackendFile_t *)backend;

    if (file_backend->fd != NULL)
        fclose(file_backend->fd);

    /* Wait until any pending message is processed. */
    pthread_join(file_backend->base.BackendThread, NULL);

    free(file_backend);
}

/*
 * \func exchnd_backend_create_file
 *
 * Create the file backend. The functions tries open the file or uses STDOUT
 * if "-" is given as filename.
 *
 * \param next Pointer to linked list of backends
 * \param filename Filename of the file this backend writes to
 *
 * \return pointer to new backend or NULL memory allocation failed
 */
ExchndBackend_t *exchnd_backend_create_file(
    ExchndBackend_t *next, char *filename)
{
    ExchndBackendFile_t *file_backend =
        calloc(1, sizeof(ExchndBackendFile_t));
    pthread_attr_t attr;

    if (file_backend != NULL) {
        int flags;
        int fd = 0;
        /* setup the exception handler backend interface */
        file_backend->base.store = store;
        file_backend->base.destroy = destroy;
        file_backend->base.next = next;

        /* filename given, output into a file */
        file_backend->fd = fopen(filename, "a+");

        if (file_backend->fd == NULL) {
            free(file_backend);
            exchnd_print_error("Could not create file %s. Error %d",
                               filename, errno);
            return next;
        }

        /* No buffering, sync on write */
        fd = fileno(file_backend->fd);

        /* Klockwork Fix */
        if (fd != -1) {
            setvbuf(file_backend->fd, NULL, _IONBF, 0);
            flags = fcntl(fd, F_GETFL) | O_DSYNC;

            fcntl(fd, F_SETFL, flags);
        }

        chmod(filename, S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH);

        pthread_attr_init(&attr);
        pthread_attr_setstacksize(&attr, PTHREAD_STACK_MIN);

        pthread_create(&file_backend->base.BackendThread,
                       &attr,
                       retrieve_msg,
                       file_backend);

        pthread_attr_destroy(&attr);

        /* We assume base to be the first element. */
        return (ExchndBackend_t *)file_backend;
    } else {
        return next;
    }
}